import React, { useState, useEffect } from "react";

interface ModalProps {
  isOpen: boolean;
  onClose: () => void;
  children: React.ReactNode;
  size?: "small" | "medium" | "large" | "screen";
  position?: "center" | "leftFull" | "rightFull" | "bottom" | "top";
  backdropClosable?: boolean;
  enterFrom?: "left" | "right" | "top" | "bottom";
  backdrop: boolean;
  modalClssName?: string;
  modalBodyClass?: string;
}

const Modal: React.FC<ModalProps> = ({
  isOpen,
  onClose,
  children,
  size = "medium",
  backdropClosable = true,
  enterFrom = "top",
  backdrop,
  position,
  modalClssName,
  modalBodyClass,
}) => {
  const [isAnimating, setIsAnimating] = useState(false);
  const [shouldRender, setShouldRender] = useState(isOpen);

  useEffect(() => {
    if (isOpen) {
      setShouldRender(true);
      setTimeout(() => setIsAnimating(true), 50);
    } else {
      setIsAnimating(false);
      setTimeout(() => setShouldRender(false), 100);
    }
  }, [isOpen]);

  if (!shouldRender) return null;

  const enterClasses = {
    left: "transform translate-x-[-100%] opacity-10",
    right: "transform -translate-x-[-100%] opacity-10",
    top: "transform translate-y-[-100%] opacity-10",
    bottom: "transform -translate-y-[-100%] opacity-10",
  };

  const enterActiveClasses = {
    left: "transform translate-x-0",
    right: "transform translate-x-0",
    top: "transform translate-y-0",
    bottom: "transform translate-y-0",
  };

  const exitClasses = {
    left: "transform -translate-x-[100%] opacity-10",
    right: "transform translate-x-[100%] opacity-10",
    top: "transform -translate-y-[100%] opacity-10",
    bottom: "transform translate-y-[100%] opacity-10",
  };

  const backdropClass = `w-screen h-screen z-[1040] flex items-center justify-center bg-gray-950/40 bg-opacity-40 fixed inset-0 transition-opacity duration-300 ${
    isAnimating ? "opacity-100" : "opacity-0"
  }`;
  const modalClass = `${modalClssName ?? ""} ${
    position === "center"
      ? `left-[50%] -translate-x-[50%] rounded-lg ${
          size === "screen" ? "top-0" : "top-20"
        }`
      : position === "leftFull"
      ? "left-0 top-0 bottom-0 w-[300px] rounded-tr-xl rounded-br-xl"
      : position === "rightFull"
      ? "right-0 top-0 bottom-0 w-[300px] rounded-tl-xl rounded-bl-xl"
      : position === "bottom"
      ? "bottom-0 right-0 left-0 top-auto bottom-0 !w-full rounded-tl-xl rounded-tr-xl"
      : position === "top"
      ? "right-0 left-0 top-0 w-full "
      : size === "screen"
      ? "top-0 left-0 right-0"
      : "top-10 left-[50%] -translate-x-[50%]"
  } bg-white p-6 pl-3  shadow-lg fixed transition-transform duration-200 z-[1050] ${
    isAnimating ? enterActiveClasses[enterFrom] : enterClasses[enterFrom]
  } ${!isOpen && !isAnimating ? exitClasses[enterFrom] : ""} ${
    size === "large"
      ? "w-11/12 md:w-2/3"
      : size === "medium"
      ? "w-11/12 sm:2/3 md:w-1/2"
      : size === "small"
      ? "w-11/12 sm:3/4 md:w-2/4 2xl:w-1/3"
      : size === "screen"
      ? "w-full h-full top-0 rounded-0 overflow-auto"
      : ""
  }`;

  return (
    <>
      {backdrop && (
        <div
          className={backdropClass}
          onClick={backdropClosable ? onClose : undefined}
        ></div>
      )}
      <div className={`${modalClass} min-w-56 min-h-28 modal rounded-2xl`}>
        <button onClick={onClose} className="absolute top-4 left-3">
          <svg
            xmlns="http://www.w3.org/2000/svg"
            fill="none"
            viewBox="0 0 24 24"
            strokeWidth={1.5}
            stroke="currentColor"
            className="size-6"
          >
            <path
              strokeLinecap="round"
              strokeLinejoin="round"
              d="M6 18 18 6M6 6l12 12"
            />
          </svg>
        </button>
        <div className={`modal-body pl-3 ${modalBodyClass}`}>{children}</div>
      </div>
    </>
  );
};

export default Modal;
