import React, { ButtonHTMLAttributes } from "react";

type ButtonProps = {
  children: React.ReactNode;
  className?: string;
  isLoading?: boolean;
  color?: "primary" | "success" | "danger" | "white" | "warning";
  isValid?: boolean;
  size?: "icon" | "md" | "sm" | "lg";
  variant?: "text" | "contained" | "outline" | "link";
} & ButtonHTMLAttributes<HTMLButtonElement>;

function Button({
  children,
  className = "",
  color = "primary",
  isValid,
  isLoading,
  variant = "contained",
  size = "md",
  ...rest
}: ButtonProps) {
  const variantClasses = {
    primary:
      variant === "contained"
        ? "bg-primary-main text-white hover:bg-primary-hover border-primary-main hover:border-primary-hover"
        : variant === "outline"
        ? "text-primary-main bg-transparent border-primary-main"
        : variant === "text"
        ? "bg-transparent text-primary-main border-0"
        : "bg-white underline text-primary-main border-white",
    danger:
      variant === "contained"
        ? "bg-error-main text-white hover:bg-error-hover border-error-main hover:border-error-hover"
        : variant === "outline"
        ? "text-red-600 bg-white border-red-600"
        : variant === "text"
        ? "bg-white text-red-600 border-white"
        : "bg-white underline text-red-600 border-white",
    success: "bg-green-500 text-white hover:bg-green-600",
    warning: "bg-warning-main text-white hover:bg-warning-hover",
    white:
      variant === "contained"
        ? "bg-white text-gray-800"
        : variant === "outline"
        ? "text-dark-blue-800 bg-white border-dark-blue-800"
        : variant === "text"
        ? "bg-white text-dark-blue-800 border-white"
        : "bg-white underline text-dark-blue-800 border-white",
  };

  const sizeClasses = {
    icon: "size-[50px]",
    sm: "h-9",
    md: "h-10",
    lg: "h-11",
  };

  const btnClass = `cursor-pointer flex justify-center border items-center gap-2 rounded-[50px] ${
    variantClasses[color]
  } ${sizeClasses[size]} ${
    isValid === false ? "opacity-50 cursor-not-allowed" : ""
  } ${className} ${size === "icon" ? "p-2" : "p-3"}`;

  return (
    <button className={btnClass} disabled={isValid === false} {...rest}>
      {isLoading ? <span className="loader"></span> : children}
    </button>
  );
}

export default Button;
